<?php
if (!defined('ABSPATH')) exit;
/**
 * Plugin Name: WEBXPAY for Woocommerce
 * Plugin URI: https://webxpay.co/
 * Description: The WEBXPAY WooCommerce Payment Gateway Plugin is your ultimate solution for lightning-fast and secure transactions. Seamlessly integrated with WooCommerce, it offers a smooth, efficient, and fully customizable checkout experience. Elevate your e-commerce platform with WEBXPAY and deliver a superior shopping experience to your customers.
 * Version: 5.4
 * Author: WEBXPAY (Private) Limited
 * Author URI: https://webxpay.co/
 * License: GPLv2 or later
 * License URI: http://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: webxpay-for-woocommerce
 * Domain Path: /languages
 *
 *
 * Copyright (c) 2017-2024 WEBXPAY
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 *
 * @package WEBXPAY
 */
use WebXPay\Blocks\Payment\WebXPay;
add_action('plugins_loaded', 'webxpay_payment_gateway', 0);

function webxpay_payment_gateway()
{
    

	if (!class_exists('WC_Payment_Gateway')) return;

	class WebXPay_Payment extends WC_Payment_Gateway
	{
		public $liveurl;
		public $secret_key;
		public $public_key;
		public $checkout_msg;
		public $msg;
		public $tokenize;

		public function __construct()
		{
			// Hooks
            $plugin_dir = plugin_dir_url(__FILE__);
			$this->id = 'webxpay';
			$this->icon = $plugin_dir . 'public/images/webxpay_logo.jpg';
			$this->method_title = __("WebXpay", 'webxpay');
			$this->has_fields = false;
            
			$this->init_form_fields();
			$this->init_settings();

			foreach ($this->settings as $setting_key => $value) {
				$this->$setting_key = $value;
			}

			$this->msg['message'] 	= "";
			$this->msg['class'] 		= "";
            
			add_action('init', array(&$this, 'check_webxpay_response'));
		
			if (version_compare(WOOCOMMERCE_VERSION, '2.0.0', '>=')) {
				add_action('woocommerce_update_options_payment_gateways_' . $this->id, array(&$this, 'process_admin_options'));
			} else {
				add_action('woocommerce_update_options_payment_gateways', array(&$this, 'process_admin_options'));
			}
			
			add_action('woocommerce_receipt_webxpay', array(&$this, 'receipt_page'));
		}

		function init_form_fields()
		{
			$this->form_fields = array(
				'enabled' => array(
					'title' => __('Enable/Disable', 'webxpay'),
					'type' => 'checkbox',
					'label' => __('Enable WebXpay Module.', 'webxpay'),
					'default' => 'no'
				),

				'title' => array(
					'title' => __('Title:', 'webxpay'),
					'type' => 'text',
					'description' => __('This controls the title which the user sees during checkout.', 'webxpay'),
					'default' => __('Visa / Master / Amex / Diners Club / Discover / China Union Pay / DFCC Wallet / Sampath Vishwa / eZcash / MCash / FriMi / UPay / Genie', 'webxpay')
				),

				'description' => array(
					'title' => __('Description:', 'webxpay'),
					'type' => 'textarea',
					'description' => __('This controls the description which the user sees during checkout.', 'webxpay'),
					'default' => __('Visa / MasterCard', 'webxpay')
				),

				'liveurl' => array(
					'title' => __('PG Domain:', 'webxpay'),
					'type' => 'text',
					'description' => __('Payment process URL', 'webxpay'),
					'default' => __('https://webxpay.com/index.php?route=checkout/billing', 'webxpay')
				),

				'secret_key' => array(
					'title' => __('Secret Key:', 'webxpay'),
					'type' => 'text',
					'description' => __('Secret key given by WebXpay.', 'webxpay'),
					'default' => __('D15AA1E2482BCD6D56E5F3A4B5DE4', 'webxpay')
				),

				'public_key' => array(
					'title' => __('Public Key:', 'webxpay'),
					'type' => 'textarea',
					'description' => __('Public key given by WebXpay.', 'webxpay'),
					'default' => __('-----BEGIN PUBLIC KEY----- MIGfMA0GCSqGSIb3DQEBAQUAA4GNADCBiQKBgQDMgKoWe7+6lkjIoYO0nr/nFxzm nzctfu0xIanhqe7s0odp4ft3wRaK6EG3UxfsRqXQQ05lW6PTIzZgaa+Dm/AYl4vy egTHkpy2M/23j5zjdv6wZapKgLq+Q6LxbJe0EWre6tdiqoY4uQurWQqgeuq0+p1H 7qIrwD+3hXiIa+KgrQIDAQAB -----END PUBLIC KEY-----', 'webxpay')
				),

				'checkout_msg' => array(
					'title' => __('Checkout Message:', 'webxpay'),
					'type' => 'textarea',
					'description' => __('Message display when checkout', 'webxpay'),
					'default' => __('Thank you for your order. Please click the button below to pay with WebXpay.', 'webxpay')
				),
				'tokenize' => array(
					'title' => __('Save Card', 'webxpay'),
					'type' => 'checkbox',
					'label' => __('Enable the option for one click payment.', 'webxpay'),
					'default' => __('yes','webxpay')
				),

			);
		}
        

		public function admin_options()
		{
			echo esc_html($this->description);
			echo '<h3>' .esc_html__('WebXpay Payment Gateway', 'webxpay') . '</h3>';
			echo '<p>' .esc_html__('WebXpay Payment Gateway allows you to accept payments from customers using Visa / MasterCard / AMEX / EzCash / Mcash / Sampath Vishwa and any new payment options made available to public use.', 'webxpay') . '</p>';
			echo '<table class="form-table">';
			$this->generate_settings_html();
			echo '</table>';
		}

		function payment_fields()
		{
			if ($this->description) echo wp_kses_post(wpautop(wptexturize($this->description)));
		}

		function receipt_page($order)
		{

			global $woocommerce;
			$order_details = new WC_Order($order);

			echo '<br>' . esc_html($this->checkout_msg) . '</b>';
			echo $this->generate_ipg_form($order);
		}

		public function generate_ipg_form($order_id)
		{
			global $woocommerce;

			$order = new WC_Order($order_id);

			//Format order number
			$formatted_order_number = $order->get_meta('_order_number_formatted');
			if (!isset($formatted_order_number) || $formatted_order_number == '') {
				$formatted_order_number = $order_id;
			}
			//End format order number

			$productinfo = "Order " . $order_id;

			$currency_code 	= get_woocommerce_currency();
			$curr_symbole 	= get_woocommerce_currency();

			// unique_order_id|total_amount|secret_key
			$plaintext = $formatted_order_number . '|' . $order->get_total();;
			//load public key for encrypting
			$publickey = $this->public_key;
			openssl_public_encrypt($plaintext, $encrypted, $publickey);

			$payment = base64_encode($encrypted);

			global $wpdb;
			$table_name = $wpdb->prefix . 'webxpay_ipg';
			$check_oder = $wpdb->prepare("SELECT COUNT(*) FROM %s WHERE merchant_reference_no = %d",$table_name,$order_id);

			if ($check_oder > 0) {
				$wpdb->update(
					$table_name,
					array(
						'order_id' => $order_id,
						'order_reference_number' => '',
						'currency_code' => $order->get_currency(),
						'amount' => $order->get_total(),
						'date_added' => gmdate('Y-m-d H:i:s'),
						'date_time_transaction' => '',
						'status_code' => '',
						'comment' => ''
					),
					array('merchant_reference_no' => $order_id)
				);
			} else {

				$wpdb->insert(
					$table_name,
					array(
						'order_id' => $order_id,
						'order_reference_number' => '',
						'currency_code' => $order->get_currency(),
						'amount' => $order->get_total(),
						'date_added' => gmdate('Y-m-d H:i:s'),
						'date_time_transaction' => '',
						'status_code' => '',
						'comment' => ''
					),
					array('%s', '%s', '%s', '%s', '%s', '%s', '%s', '%s', '%s', '%d')
				);
			}

			//Get customer details
			$first_name = $order->get_billing_first_name();
			$last_name = $order->get_billing_last_name();
			$email = $order->get_billing_email();
			$contact_number = $order->get_billing_phone();
			$address_line_one = $order->get_shipping_address_1();
			$address_line_two = $order->get_shipping_address_2();
			$city = $order->get_shipping_city();
			$state = $order->get_shipping_state();
			$postal_code = $order->get_shipping_postcode();

			$country = $order->get_shipping_country();
			//create nones
			$my_nonce = wp_create_nonce('my_nonce-'.$order_id);


			$tokenize = $this->tokenize;
			
			if(is_user_logged_in() && $tokenize !='no'){
				$user = wp_get_current_user();
				$timestamp = time();
				
				$customer_info_text = $user->ID.'|'.$user->user_email.'|'.$timestamp;
				
				//load public key for encrypting
				openssl_public_encrypt($customer_info_text, $encrypt_customer_info, $this->public_key);
		
				//encode for data passing
				$customer_info = base64_encode($encrypt_customer_info);
				
				$form_args = array(
					'secret_key' => $this->secret_key,
					'payment' => $payment,
					'first_name' => (!empty($first_name) ? $first_name : ''),
					'last_name' => (!empty($last_name) ? $last_name : ''),
					'token_parameter' => $customer_info,
					'is_tokenize' => true,
					'version' => '5.2',
					'email' => (!empty($email) ? $email : ''),
					'contact_number' => (!empty($contact_number) ? $contact_number : ''),
					'address_line_one' => (!empty($address_line_one) ? $address_line_one : ''),
					'address_line_two' => (!empty($address_line_two) ? $address_line_two : ''),
					'city' => (!empty($city) ? $city : ''),
					'state' => (!empty($state) ? $state : ''),
					'postal_code' => (!empty($postal_code) ? $postal_code : ''),
					'country' => (!empty($country) ? $country : ''),
					'process_currency' => get_woocommerce_currency(),
					'cms' => 'WordPress-4.1',
					'custom_fields' => $my_nonce,
					'enc_method' => 'JCs3J+6oSz4V0LgE0zi/Bg==',
				);
				
			}else{
				$form_args = array(
					'secret_key' => $this->secret_key,
					'payment' => $payment,
					'first_name' => (!empty($first_name) ? $first_name : ''),
					'last_name' => (!empty($last_name) ? $last_name : ''),
					'email' => (!empty($email) ? $email : ''),
					'contact_number' => (!empty($contact_number) ? $contact_number : ''),
					'address_line_one' => (!empty($address_line_one) ? $address_line_one : ''),
					'address_line_two' => (!empty($address_line_two) ? $address_line_two : ''),
					'city' => (!empty($city) ? $city : ''),
					'state' => (!empty($state) ? $state : ''),
					'postal_code' => (!empty($postal_code) ? $postal_code : ''),
					'country' => (!empty($country) ? $country : ''),
					'process_currency' => get_woocommerce_currency(),
					'cms' => 'WordPress-4.1',
					'custom_fields' => $my_nonce,
					'enc_method' => 'JCs3J+6oSz4V0LgE0zi/Bg==',
				);
			}
			
			//create form array
			

			$form_args_array = array();
			foreach ($form_args as $key => $value) {
				$form_args_array[] = "<input type='hidden' name='$key' value='$value'/>";
			}
			include('includes/webxpay-form.php'); 
			
		}

		function process_payment($order_id)
		{

			$order = new WC_Order($order_id);
			return array(
				'result' => 'success', 'redirect' => add_query_arg(
					'order-pay',
					$order->id,
					add_query_arg('key', $order->order_key, get_permalink(woocommerce_get_page_id('pay')))
				)
			);
		}

		function check_webxpay_response()
		{
			global $woocommerce;
			global $wpdb;
			global $webxpay_meta_value;

			if (isset($_POST['payment']) && isset($_POST['signature']) && isset($_POST['custom_fields']) && wp_verify_nonce(sanitize_text_field($_POST['custom_fields']), 'my_nonce-'.sanitize_text_field($_POST['order_id']))) {

				$payment = base64_decode(sanitize_text_field($_POST["payment"]));
				$signature = base64_decode(sanitize_text_field($_POST["signature"]));

				$publickey = $this->public_key;

				openssl_public_decrypt($signature, $value, $publickey);

				$signature_status = false;

				if ($value == $payment) {
					$signature_status = true;
				}

				//proceed if signatures match only
				if ($signature_status) {

					//get payment response in segments
					//payment format: order_id|order_refference_number|date_time_transaction|payment_gateway_id|status_code|comment;
					$responseVariables = explode('|', $payment);


					if (!empty($responseVariables)) {
						//Get order ID from formatted
						$tbl = $wpdb->prefix . 'postmeta';
						//$prepare_guery = $wpdb->prepare("SELECT post_id FROM %s where meta_key ='_order_number_formatted' AND meta_value = %d ",$tbl, $webxpay_meta_value);
						$get_values = $wpdb->get_col($wpdb->prepare("SELECT post_id FROM %s where meta_key ='_order_number_formatted' AND meta_value = %d ",$tbl, $webxpay_meta_value));

						if (isset($get_values) && $get_values[0] != '') {
							$order_id = $get_values[0];
						} else {
							$order_id = $responseVariables[0];
						}

						$order_reference_number = $responseVariables[1];
						$date_time_transaction = $responseVariables[2];
						$response_code = $responseVariables[3];
						$status_code = $responseVariables[4];
						$comment = $responseVariables[5];
						$payment_gateway_id = $responseVariables[5];
						$order 	= new WC_Order($order_id);

						//update database							
						$table_name = $wpdb->prefix . 'webxpay_ipg';
						$wpdb->update(
							$table_name,
							array(
								'order_id' => $order_id,
								'order_reference_number' => $order_reference_number,
								'currency_code' => $order->get_currency(),
								'date_time_transaction' => $date_time_transaction,
								'payment_gateway_id' => $payment_gateway_id,
								'status_code' => $status_code,
								'comment' => $comment
							),
							array('order_id' => $order_id)
						);

						if ($responseVariables[3] == '10' || $responseVariables[3] == '00') {

							$order->add_order_note('WebXpay Payment successful<br/>Order reference number: ' . $order_reference_number);
							$order->add_order_note($this->msg['message']);

							$additional_fee_discount_message = sanitize_text_field($_POST["additional_fee_discount_message"]);
							//$this->custom_wc_add_fee($_POST, $order, $responseVariables);
							$this->custom_add_fee($additional_fee_discount_message, $order, $responseVariables);


							$order->payment_complete();
							$woocommerce->cart->empty_cart();
							wp_redirect($order->get_checkout_order_received_url());
							exit();
						} else {
							$order->update_status('failed');
							$order->add_order_note('WebXpay Payment unsuccessful<br/>Order reference number: ' . $order_reference_number . '<br/>Status code: ' . $status_code . '<br/>Comment: ' . $comment . '');
							$order->add_order_note($this->msg['message']);

							$woocommerce->cart->empty_cart();
							wp_redirect($order->get_checkout_order_received_url());
							exit();
						}
					}
				}
			}
		}


		function custom_add_fee($additional_fee_discount_message, $order, $responseVariables)
		{
			//$additional_fee_discount_message = $post["additional_fee_discount_message"];
			$requested_amount = $responseVariables[6];

			$transaction_amount = $responseVariables[7];
			$order_amount = $order->get_total();

			if ($transaction_amount != $order_amount) {

				$total_different = $transaction_amount - $order_amount;

				// Get the customer country code
				$country_code = $order->get_shipping_country();

				// Set the array for tax calculations
				$calculate_tax_for = array(
					'country' => $country_code,
					'state' => '',
					'postcode' => '',
					'city' => ''
				);

				// Get a new instance of the WC_Order_Item_Fee Object
				$item_fee = new WC_Order_Item_Fee();

				$imported_total_fee = $total_different;

				$item_fee->set_name($additional_fee_discount_message); // Generic fee name
				$item_fee->set_amount($imported_total_fee); // Fee amount
				$item_fee->set_tax_class(''); // default for ''
				$item_fee->set_tax_status('taxable'); // or 'none'
				$item_fee->set_total($imported_total_fee); // Fee amount

				// Calculating Fee taxes
				$item_fee->calculate_taxes($calculate_tax_for);

				// Add Fee item to the order
				$order->add_item($item_fee);

				$order->calculate_totals();
				$order->save();
			}
		}

		function webxpay_custom_redirect_after_purchase()
		{
			//return '<p>I am Here!</p>';
		}

		function get_pages($title = false, $indent = true)
		{
			$wp_pages = get_pages('sort_column=menu_order');
			$page_list = array();
			if ($title) $page_list[] = $title;
			foreach ($wp_pages as $page) {
				$prefix = '';
				if ($indent) {
					$has_parent = $page->post_parent;
					while ($has_parent) {
						$prefix .=  ' - ';
						$next_page = get_page($has_parent);
						$has_parent = $next_page->post_parent;
					}
				}
				$page_list[$page->ID] = $prefix . $page->post_title;
			}
			return $page_list;
		}
	}
    require_once dirname( __FILE__ ) . '/webxpay_payment_register.php';
    // Instantiate the WCWEBXPAY_GATEWAY_WEB class
    $webxpay_gateway = new WEBXPAY_GATEWAY();
    $webxpay_gateway->init_hooks();
	
	// Catch payment response
	if (isset($_POST['payment']) && isset($_POST['signature']) && wp_verify_nonce(sanitize_text_field($_POST['custom_fields']), 'my_nonce-'.sanitize_text_field($_POST['order_id']))) {
			$WC = new WebXPay_Payment();
	}

	function webxpay_add($methods)
	{
		$methods[] = 'WebXPay_Payment';
		return $methods;
	}

	add_filter('woocommerce_payment_gateways', 'webxpay_add');
}



global $webxpay_db_version;
$webxpay_db_version = '1.0';

function webxpay_install()
{
	global $wpdb;
	global $webxpay_db_version;

	$table_name = $wpdb->prefix . 'webxpay_ipg';
	$charset_collate = '';

	if (!empty($wpdb->charset)) {
		$charset_collate = "DEFAULT CHARACTER SET {$wpdb->charset}";
	}

	if (!empty($wpdb->collate)) {
		$charset_collate .= " COLLATE {$wpdb->collate}";
	}

	$sql = "CREATE TABLE $table_name (
				id int(9) NOT NULL AUTO_INCREMENT,
				order_id int(9) NOT NULL,
				order_reference_number VARCHAR(20) DEFAULT NULL,					
				currency_code VARCHAR(6) NOT NULL,
				amount VARCHAR(20) NOT NULL,					
				date_added datetime NOT NULL,
				date_time_transaction datetime DEFAULT '0000-00-00 00:00:00',
				payment_gateway_id int(5) DEFAULT 0,
				status_code VARCHAR(10) DEFAULT NULL,
				comment VARCHAR(100) DEFAULT NULL,
				merchant_reference_no VARCHAR(20) DEFAULT NULL,
				UNIQUE KEY id (id)
			) CHARSET=utf8;";


	require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
	dbDelta($sql);

	add_option('webxpay_db_version', $webxpay_db_version);
}

function webxpay_install_data()
{
	global $wpdb;

	$welcome_name = 'WebXpay';
	$welcome_text = 'Congratulations, you just completed the installation!';

	$table_name = $wpdb->prefix . 'webxpay_ipg';

	$wpdb->insert(
		$table_name,
		array(
			'time' => current_time('mysql'),
			'name' => $welcome_name,
			'text' => $welcome_text,
		)
	);
}

register_activation_hook(__FILE__, 'webxpay_install');
register_activation_hook(__FILE__, 'webxpay_install_data');